/* 
   orbit - calculate Charon's orbit
*/

#include <stdio.h>
#include <malloc.h>
#include <math.h>
#define PI M_PI

#define AU 1.49597870e8    /* astronomical unit in km,
                              from the Astronomical Almanac 1992 */
#define EDIST0 (38.5 * AU)  /* Pluto's mean opposition earth distance */
#define SDIST0 (39.5 * AU)  /* Pluto's mean opposition sun distance */
#define SECPERDAY (24*3600)
#define LIGHTSPEED ( 299792.458)  /* km per sec */
#define DPR 57.295779513   /* degrees per radian */
#define HPR (DPR/15.0)     /* hours per radian */
#define APR (DPR * 3600.0)     /* arcsec per radian */
#define NEW(x,y) ((x *) malloc(y * sizeof(x)))

#define USAGE "magnitude jd0 a e i Omega longperi L P < eph > magfile"
double from_base60(char *b);
double mod2pi(double d);

#define JD0       2449000.5
#define SEMIMAJ   19636.0
#define ECC       0.0076
#define INCL      96.093
#define ASCNODE   223.000
#define LONGPERI  219.2
#define MEANLONG0 32.896
#define PERIOD    6.387221

main(int argc, char *argv[])
{
  double jd0, semimaj, ecc, incl, ascnode, longperi, meanlong0, period;
  char s[240];

  /* Get the arguments */
  if (argc < 9 ) {
    if (argc == 1) {
      jd0 = JD0;  semimaj = SEMIMAJ; ecc = ECC; incl = INCL;
      ascnode = ASCNODE; longperi = LONGPERI; 
      meanlong0 = MEANLONG0; period = PERIOD;
    }
    else {
      printf("%s\n", USAGE);
      exit(1);
    }
  }
  else {
    jd0 = atof(argv[1]);  
    semimaj = atof(argv[2]); 
    ecc = atof(argv[3]); 
    incl = atof(argv[4]);
    ascnode = atof(argv[5]); 
    longperi = atof(argv[6]); 
    meanlong0 = atof(argv[7]); 
    period = atof(argv[8]);
  }

  /* turn degrees into radians */
  incl /= DPR;
  ascnode /= DPR;
  longperi /= DPR;
  meanlong0 /= DPR;

  printf("#%-10s Mc-Mp Mc-Ms Mp-Ms  Mp    Mc    Ms   long\n", "id");
  while (fgets(s, 254, stdin) != NULL) {
      do_line(s, 
	      jd0, semimaj, ecc, incl, ascnode, longperi, meanlong0, period);
    }
}

do_line(char s[256], 
	double jd0, double semimaj, double ecc, double incl, 
        double ascnode, double longperi, double meanlong0, double period)
{
  int nscan;
  double jd, ra, dec, dist, sdist;
  char id[80], rastr[80], decstr[80];
  double meananom, argperi;
  double dec1, ra1, num, denom, k, subElong;
  double a, pmag, cmag, smag;

  if (s[0] == '#') {
    printf(s);
    return ;
  }

  /* ra and dec of pole */
  ra1 = ascnode - PI/2;  
  dec1 = PI/2 - incl; 
  argperi = longperi-ascnode;

  nscan = sscanf(s, "%s %*s %lf %*s %s %s %lf", id, &jd, rastr, decstr, &dist);
  if (nscan != 5) {
    return ;
  }
  
  ra = from_base60(rastr)/HPR;
  dec = from_base60(decstr)/DPR;
  dist *= AU;
  
  /* mean anomoly */
  meananom = meanlong0 - longperi + 
      (2*PI/period) * ( jd-(dist/LIGHTSPEED)/SECPERDAY-jd0 );
  
  /* subElong */
  num = -cos(dec1)*sin(dec) + sin(dec1)*cos(dec)*cos(ra1-ra);
  denom = cos(dec)*sin(ra1-ra);
  k = atan2(num, denom);
  subElong = mod2pi(-((meananom+argperi) - k));

  /* solar phase angle */
  a = -1.9041 + 0.00357 * (jd - 2449770.5);
  sdist = y = 29.826 + 0.00018766 * (jd - 2449770.5);

  /* magnitudes */
  pmag = plutomag(subElong, a, dist, sdist);
  cmag = charonmag(subElong, a, dist, sdist);
  smag = -2.5 * log10( pow(10., -0.4*pmag) + pow(10., -0.4*cmag) )
  
  printf("%-11s %5.2f %5.2f %5.2f %5.2f %5.2f %5.2f %5.2f\n",
     id, cmag-pmag, cmag-smag, pmag-smag, pmag, cmag, smag, subElong * DPR);
}    

double plutomag(double l, double a, double dist, double sdist)
{
double s, c, m;

  m = 15.3735 + 0.0294 * a;
  m +=  0.0258 * cos(l)     + 0.1284 * sin(l);
  m += -0.0398 * cos(2 * l) - 0.0348 * sin(2 * l);
  m +=  0.0050 * cos(3 * l) - 0.0012 * sin(3 * l);
  m +=  0.0054 * cos(4 * l) - 0.0019 * sin(4 * l);
  return (m + 5 * log10(dist/EDIST0) + 5 * log10(sdist/SDIST0) );
}

double charonmag(double l, double a, double dist, double sdist)
{
double s, c, m, sdist;

  m = 17.2591 + 0.0866 * a;
  m += -0.0334 * cos(l)     + 0.0005 * sin(l);
  m +=  0.0148 * cos(2 * l) - 0.0083 * sin(2 * l);
  return (m);
}

double mod2pi(double d)
{
 return (d - 2*PI*floor(d/(2*PI)));
}

double from_base60(char *b)
{
int p, d, m; /* plus/minus, degree, minute */
double s;    /* seconds */
int i, j;       /*index */
char n[15];
  
  /* p - if the first non-white char is -, p = -1 */
  i = 0;
  while (isspace(b[i]))   i++;
  if (b[i] == '-')
  {
    p = -1;
    i++;
  }
  else
    p = 1;

  /* d - skip white space, decode a number, skip white, skip colon */
  while (isspace(b[i]))   i++;
  j = 0;
  while (isdigit(b[i]))
    n[j++] = b[i++];
  n[j]='\0';
  sscanf(n, "%d", &d);
  while (isspace(b[i]))   i++;
  if (b[i] == ':') i++;

  /* m - skip white, decode number, skip white, skip colon */
  while (isspace(b[i]))   i++;
  j = 0;
  while (isdigit(b[i]))
    n[j++] = b[i++];
  n[j]='\0';
  sscanf(n, "%d", &m);
  while (isspace(b[i]))   i++;
  if (b[i] == ':') i++;

  /* ds -skip white, decode a number */
  while (isspace(b[i]))   i++;
  j = 0;
  while (isdigit(b[i]) || b[i] == '.')
    n[j++] = b[i++];
  n[j]='\0';
  sscanf(n, "%lf", &s);
  while (isspace(b[i]))   i++;
/*printf("%s -> %d * %d:%d:%f\n", b, p, d, m, s);*/
  return( p * (d + m/60. + s/3600.) );
}


