;+
; Return the fractional projected area
;
; Map array should be 2 dimensional [lon,lat].
;
; Map values should be in the range zero to one (1=full coverage, 0=none).
;
; Left edge (x=0) is zero longitude, with higher values progressing East.
;
; Top edge (y=0) is North because most plotting programs display y=0 at
; the top (but beware that IDL does the opposite).  If the map has South
; on top as shown by a typical plotting program (xv, gimp, display, etc.),
; invert the sense by setting either the /south or /north keyword (doesn't
; matter which).
;
; Use the grid keyword to specify lat and lon grid spacing (degrees),
; for a wire grid to be superposed on the map.
;
; Keywords:
;    lat	Sub-viewer latitudes (degrees) (default = 0)
;    lon	Sub-viewer longitudes (degrees) (default = 0)
;    frac	Return fractional projected coverage (full coverage is 1.0)
;-

FUNCTION	map_da,map_in,lat=in_lat,lon=in_lon, $
        frac=frac,north=north,south=south

    lat = [0.0]
    lon = [0.,90.,180.,270.,360.]
    IF n_elements(in_lat) gt 0 THEN lat = [in_lat]
    IF n_elements(in_lon) gt 0 THEN lon = [in_lon]
    nlats = n_elements(lat)
    nview = n_elements(lon)     ;& stop
    scale = 4.0/nview < 0.9
    IF nview ne nlats THEN BEGIN
        IF nlats eq 1 THEN lat = replicate(lat[0],nview) ELSE BEGIN
            print,'ERROR: Dimensionality of lat and lon should match.'
            return, -1
        ENDELSE
    ENDIF

    lons = lon*!PI/180.0	; convert to radians
    lats = lat*!PI/180.0
    
                                ; Default is for y=0 to be South
    map = map_in
    IF keyword_set(north) OR keyword_set(south) THEN map = reverse(map_in,2)
    sinfo = size(map)
    nx = sinfo[1]
    ny = sinfo[2]
    
    da_out = fltarr(nx, ny, nview)
    
    frac = fltarr(nview)

    ; Now calculate the projected areas
    cda = 2.0*!PI/(nx*ny)
    llat = (0.5*ny - (findgen(ny) + 0.5 ) ) * !PI/ny  ; the array of latitudes on the map
    sinllat = replicate(1,nx) # sin(llat)  ; MATRIX of sin(lat)
    cosllat = replicate(1,nx) # cos(llat)  ; MATRIX of cos(lat)
    llon = (findgen(nx)+0.5) *!PI*2.0/nx # replicate(1,ny) ; the MATRIX of longitudes on the map
    FOR i=0,nview-1 DO BEGIN

        mu = cos(llon-lons[i])*cosllat*cos(lats[i]) + sinllat*sin(lats[i])
        da = (mu > 0) * cosllat * cda
        frac[i] = total(da * map)
        da_out[*,*,i] = reverse(da,2)
    ENDFOR

    return, da_out

END
