;+
; Plot a cylindrical map and a row of orthographic projections under it.
;
; Map array should be 2 dimensional [lon,lat].
;
; Map values should be in the range zero to one (1=full coverage, 0=none).
;
; Left edge (x=0) is zero longitude, with higher values progressing East.
;
; Top edge (y=0) is North because most plotting programs display y=0 at
; the top (but beware that IDL does the opposite).  If the map has South
; on top as shown by a typical plotting program (xv, gimp, display, etc.),
; invert the sense by setting either the /south or /north keyword (doesn't
; matter which).
;
; Use the grid keyword to specify lat and lon grid spacing (degrees),
; for a wire grid to be superposed on the map.
;
; Keywords:
;    lat	Sub-viewer latitudes (degrees) (default = 0)
;    lon	Sub-viewer longitudes (degrees) (default = 0)
;    frac	Return fractional projected coverage (full coverage is 1.0)
;-

PRO	map_proj,map_in,lat=in_lat,lon=in_lon, $
        frac=frac,north=north,south=south,noplot=noplot,grid=grid, $
	noerase=noerase, line=line, noortho=noortho

    ; set up to plot correctly
    dsave = !d
    xs = 796 & ys = 580
    if (!d.window eq -1) then begin
        window, 0, xs = xs, ys = ys
    end
    if !d.x_size ne xs or !d.y_size ne ys then begin
        window, !d.window, xs = xs, ys = ys
    endif
    device, true_color=24, decompose = 0

    lat = [0.0]
    lon = [0.,90.,180.,270.,360.]
    IF n_elements(in_lat) gt 0 THEN lat = [in_lat]
    IF n_elements(in_lon) gt 0 THEN lon = [in_lon]
    nlats = n_elements(lat)
    nview = n_elements(lon) ;& stop
    scale = 4.0/nview < 0.9
    IF nview ne nlats THEN BEGIN
	IF nlats eq 1 THEN lat = replicate(lat[0],nview) ELSE BEGIN
	    print,'ERROR: Dimensionality of lat and lon should match.'
	    return
	ENDELSE
    ENDIF

    lons = lon*!PI/180.0	; convert to radians
    lats = lat*!PI/180.0

    IF not keyword_set(noplot) THEN $
	plot,[0],[0],/nodata,xsty=5,ysty=5,xr=[-4.5,4.5],yr=[-3.3,3.3], $
	    noerase=noerase
    ; Default is for y=0 to be South
    map = map_in
    IF keyword_set(north) OR keyword_set(south) THEN map = reverse(map_in,2)
    sinfo = size(map)
    nx = sinfo[1]
    ny = sinfo[2]

    ; First do the cylindrical map up top
    xsc = 7./nx
    ysc = 3.5/ny
    csc = 255./max(map)		; Colors run to 255
    xpat = [0,0,1,1]		; left edge is 0 lon (0th pixel is PI/nx lon)
    ypat = [1,0,0,1]
    IF not keyword_set(noplot) THEN BEGIN
	FOR x=0,nx-1 DO FOR y=0,ny-1 DO BEGIN
	    polyfill,(x+xpat)*xsc - 3.5,3.-(y+ypat)*ysc,color=map[x,y]*csc
	ENDFOR
    ENDIF
    IF not keyword_set(noplot) THEN BEGIN
	oplot,[-1,1,1,-1,-1]*3.5,[1,1,-1,-1,1]*1.75+1.25
	IF keyword_set(grid) THEN BEGIN
	    lon = 0.0
	    WHILE lon lt 360. DO BEGIN
		oplot,[lon,lon]*xsc-3.5,1.25+[-90,90]*ysc, $
                       line = line
		lon = lon + grid[0]
	    ENDWHILE
	    lat = 0.0
	    WHILE lat lt 90.0 DO BEGIN
		oplot,[0,360]*xsc-3.5,1.25+[lat,lat]*ysc, $
                       line = line
		oplot,[0,360]*xsc-3.5,1.25-[lat,lat]*ysc, $
                       line = line
		lat = lat + grid[1]
	    ENDWHILE
	ENDIF
    ENDIF

    if not keyword_set(noortho) then begin
    fraction = fltarr(nview)
    ; Now do a series of orthographic projections
    cda = 2.0*!PI/(nx*ny)
    FOR i=0,nview-1 DO BEGIN
	cov = 0.0
	posx = lons[i]*3.5/!PI - 3.5
	FOR x=0,nx-1 DO FOR y=0,ny-1 DO BEGIN

	    llat = (0.5*ny - [y,y,y+1.,y+1.,y])*!PI/ny
	    llon = ([x,x+1.,x+1.,x,x])*!PI*2.0/nx
	    da = cos((llat[0]+llat[2])*0.5)*cda

	    x0 = sin(llon)*cos(llat)
	    y0 = -cos(llon)*cos(llat)
	    z0 = sin(llat)
	    x1 = x0*cos(lons[i]) + y0*sin(lons[i])
	    y1 = - x0*sin(lons[i]) + y0*cos(lons[i])
	    z1 = z0
	    x2 = x1
	    y2 = y1*sin(lats[i]) + z1*cos(lats[i])
	    z2 = y1*cos(lats[i]) - z1*sin(lats[i])

	    xx = x2*scale + posx
	    yy = y2*scale - 1.5
	    mu = -total(z2[0:3])/4.0
	    IF mu gt 0 THEN BEGIN
		IF not keyword_set(noplot) THEN $
		    polyfill,xx,yy,color=map[x,y]*csc
		cov = cov + map[x,y]*mu*da
	    ENDIF
	ENDFOR
	theta = findgen(100)*0.069
	IF not keyword_set(noplot) THEN $
	    oplot,scale*cos(theta) + posx,scale*sin(theta) - 1.5
	IF keyword_set(frac) THEN print,string(format='(a,f5.1,a,f5.3)', $
	    'lon = ',lons[i]*180./!PI,'	cov = ',cov)
	; Optional latitude and longitude grid overlay
	IF not keyword_set(noplot) AND keyword_set(grid) THEN BEGIN
	    theta = (findgen(1001)-500.)*!PI/1000.
	    lon = 0.0
	    WHILE lon lt 360. DO BEGIN	; Draw longitude lines
		x0 = sin(lon*!PI/180.)*cos(theta)
		y0 = -cos(lon*!PI/180.)*cos(theta)
		z0 = sin(theta)
		x1 = x0*cos(lons[i]) + y0*sin(lons[i])
		y1 = - x0*sin(lons[i]) + y0*cos(lons[i])
		z1 = z0
		x2 = x1
		y2 = y1*sin(lats[i]) + z1*cos(lats[i])
		z2 = y1*cos(lats[i]) - z1*sin(lats[i])
		FOR j = 0,999 DO IF z2[j] lt 0 AND z2[j+1] lt 0 THEN $
		    oplot,posx + x2[j:j+1]*scale,y2[j:j+1]*scale - 1.5, $
                       line = line
		lon = lon + grid[0]
	    ENDWHILE
	    theta = findgen(1001)*!PI/500.
	    lat = 0.0
	    WHILE lat lt 90.0 DO BEGIN	; Draw latitude lines
		x0 = sin(theta)*cos(lat*!PI/180.)
		y0 = -cos(theta)*cos(lat*!PI/180.)
		z0 = sin(lat*!PI/180.)
		x1 = x0*cos(lons[i]) + y0*sin(lons[i])
		y1 = - x0*sin(lons[i]) + y0*cos(lons[i])
		z1 = z0
		x2 = x1
		y2 = y1*sin(lats[i]) + z1*cos(lats[i])
		z2 = y1*cos(lats[i]) - z1*sin(lats[i])
		z = where(z2 lt 0,count)
		FOR j = 0,999 DO BEGIN
		    IF z2[j] lt 0 AND z2[j+1] lt 0 THEN $
			oplot,posx + x2[j:j+1]*scale,y2[j:j+1]*scale - 1.5, $
                       line = line
		    IF z2[j] gt 0 AND z2[j+1] gt 0 THEN $
			oplot,posx + x2[j:j+1]*scale,-y2[j:j+1]*scale - 1.5, $
                       line = line
		ENDFOR
		lat = lat + grid[1]
	    ENDWHILE
	ENDIF
	fraction[i] = cov
    ENDFOR
    frac = fraction
    endif ; noortho
END
