; NAME:
;  mkmap_sinelat
; PURPOSE: (one line)
;  Make a map with a sinesoidal latitude boundary
; CALLING SEQUENCE:
;  map = mkmap_sinelat(maxlon, maxlat, minlon) 
; INPUTS:
;  maxlon - longitude of boundary maximum
;  maxlat - latitude of boundary maximum
;  minlat - latitude of boundary minimum
; OUTPUTS:
;  map - 
;   360 by 180.  Pixels are 1-deg wide.
;   Pixel 0,0 runs from latitude -90 to -89 and East longitude 0 to 1.
;  lat_deg - the latitude in degrees (360 x 180 array)
;  lon_deg - the East longitude in degrees (360 x 180 )
; COMMON BLOCKS:
;  None
; RESTRICTIONS:
;  sslat must be -45 or -50
; MODIFICATION HISTORY:
;  Written 2007 June 26, by Leslie Young, SwRI
;-

function mkmap_sinelat, maxlon_u, maxlat_u, minlat_u, maxlon_l, maxlat_l, minlat_l

    map = fltarr(360,180)

    deg = !pi/180
    lon = findgen(360) + 0.5
    lat = arrgen(-89.5, 89.5, 1.)

    ; upper boundary limit
    lat_mean_u = (maxlat_u + minlat_u)/2.
    lat_ampl_u = (maxlat_u - minlat_u)/2.
    lon_limit_u = lat_mean_u + lat_ampl_u * cos( (lon - maxlon_u) *deg)
    
    if n_params() gt 3 then begin
                                ; lower boundary limit
        lat_mean_l = (maxlat_l + minlat_l)/2.
        lat_ampl_l = (maxlat_l - minlat_l)/2.
        lon_limit_l = lat_mean_l + lat_ampl_l * cos( (lon - maxlon_l) *deg)
    endif else begin
        lon_limit_l = replicate(-90,360)
    endelse

    for i = 0, 359 do begin
        allin = where(lat lt floor(lon_limit_u[i]) and $
                      lat gt ceil(lon_limit_l[i]), nallin)
        if nallin gt 0 then map[i,allin] = 1
    endfor

    return, map

end

