; look at the observed or extrapolated surface pressures
; for 1989 (Voyager) and 1995-1997 (stellar occultation)
;
; REFERENCES
; Elliot, J. L., and 17 colleagues 2000. The Prediction and Observation of 
; the 1997 July 18 Stellar Occultation by Triton: More Evidence for 
; Distortion and Increasing Pressure in Triton's Atmosphere. Icarus 148, 
; 347-369 
;
; Elliot, J. L., D. F. Strobel, X. Zhu, J. A. Stansberry, L. H. Wasserman, 
; and O. G. Franz 2000. NOTE: The Thermal Structure of Triton's Middle 
; Atmosphere. Icarus 143, 425-428 
;
;  Gurrola, E. M. 1995. Interpretation of Radar Data from the Icy Galilean
;  Satelites and Triton. Ph.D. Thesis, Stanford University.
;
;  ~/reference/standard_atm/Triton/Triton-Yelle
;  Model atmosphere from Roger Yelle, Sept 13, 2000.  
;

pro obs_temp

    naifinit
    utc = ['1989 08 25','1995 08 14', '1997 07 18', '1997 11 04', '2004 08 07']
    yrstr = strmid(utc,0,4)
    year = float(yrstr) + (utc2et(utc) - utc2et( yrstr + ' 01 01'))/cspice_jyear()

    n = n_elements(utc) - 1  ; number of observations
    yobs = year[0:n-1]
   
    p=fltarr(n)
    perr = fltarr(n)
    
    ; Voyager
    p[0] = 14.
    perr[0] = 1.

    ; Occultations
    ratio = 19.0/2.15
    p1400 =   [1.49, (2.23 + 2.45)/2, 2.15]
    p1400err= [0.14, sqrt(0.28^2 + 0.32^2)/2, 0.02]
    p[1:n-1] = p1400 * ratio
    perr[1:n-1] = p1400err * ratio

    tobs = fltarr(n)
    tobserr = fltarr(n)
    for i=0,n-1 do begin
        tobs[i] = n2t(p[i])
        tobserr[i] = (n2t(p[i]+perr[i]) - n2t(p[i]-perr[i]))/2.
    endfor

;    ploterror, year, p, perr, line=0, xr=[1989,2005]
    ploterror, yobs, tobs, tobserr, line=0, xr=[1988,2007], xs=1,ps=4, $
      yr = [37,42], ys=3

    ; now = 1997
    tconst = tobs[n-1]
    oplot, year[n-1:n], [tobs[n-1], tconst]
    oplot, year[[n]], [[tconst]], ps=2

    ; now = linear fit to occultation temperatures (except 1993)
    tlinocccoef = poly_fit(yobs[1:n-1], tobs[1:n-1], 1,meas=tobserr[1:n-1], $
                       yfit = tlinocc)
    yplot = arrgen(yobs[1], year[n], 0.25)
    tplot = poly(yplot, tlinocccoef)
    tlinocc = max(tplot)
    oplot, yplot, tplot, line=1
    oplot, year[[n]], [[tlinocc]], ps=2
  
    ; now = quadratic fit to all temperatures (except 1993)
    tquadcoef = poly_fit(yobs, tobs, 2,meas=tobserr, $
                       yfit = tquad)
    yplot = arrgen(yobs[0], year[n], 0.25)
    tplot = poly(yplot, tquadcoef)
    tquad = max(tplot)
    oplot, yplot, tplot, line=3
    oplot, year[[n]], [[tquad]], ps=2
  
    ; now = linear fit to all temperatures (except 1993)
    tlincoef = poly_fit(yobs, tobs, 1,meas=tobserr, $
                       yfit = tlin)
    yplot = arrgen(yobs[0], year[n], 0.25)
    tplot = poly(yplot, tlincoef)
    tlin = max(tplot)
    oplot, yplot, tplot, line=5
    oplot, year[[n]], [[tlin]], ps=2
  
    ; now = linear fit to just Voyager and 1997
    tlin2coef = [0., (tobs[n-1]-tobs[0])/(yobs[n-1]-yobs[0]) ]
    tlin2coef[0] = tobs[0] - tlin2coef[1] * yobs[0]
    yplot = arrgen(yobs[0], year[n], 0.25)
    tplot = poly(yplot, tlin2coef)
    tlin2 = max(tplot)
    oplot, yplot, tplot, line=2 
    oplot, year[[n]], [[tlin2]], ps=2

    print, tlin
; 38.5
;    stop

end
