; NAME:
;  rdmap_geol
; PURPOSE: (one line)
;  Read the Grundy Young 2004 N2 map of Triton
; CALLING SEQUENCE:
;  map_geol = rdmap_geol(minlat, units, lat_deg, lon_deg)
; INPUTS:
;  minlat latitude cutoff for units 1-6 (deg)
;  unit 7-element array of fraction of unit to include
;       usually 0 for exclusion and 1 for inclusion
; KEYWORD INPUTS
;  minlat0 - latitude cutoff for unit0
; OUTPUTS:
;  map - 
;   360 by 180.  Pixels are 1-deg wide.
;   Pixel 0,0 runs from latitude -90 to -89 and East longitude 0 to 1.
;  lat_deg - the latitude in degrees (360 x 180 array)
;  lon_deg - the East longitude in degrees (360 x 180 )
; COMMON BLOCKS:
;  None
; RESTRICTIONS:
;  sslat must be -45 or -40
; MODIFICATION HISTORY:
;  Written 2007 June 26, by Leslie Young, SwRI
;  2008 May 8, LAY
;-

function rdmap_geol, minlat, unit, lat_deg, lon_deg, minlat0=minlat0


; maps are all 360 x 180, with edge of the 0,0 pixel
; at 0 East longitude, -90 latitude

    physconstants               ; set up !phys.sigma, etc
    deg = !dpi/180.d
    
    ; read in the geologic map
    geomap = rdmap_triton_mcewen90(lat_deg, lon_deg)
                                ; geomap : 7 planes of geologic units
                                ;   0 = unmapped, 1-6 = units 1-6
                                ;   value = 0 if not in unit, 1 if in unit

    latmap = lat_deg * deg
    lonmap = lon_deg * deg

    ; spcmap : south polar cap (units 2+3+5)
    ;   value = 0 if not in units, 1 if in units
    spcmap = float(unit[0]) * geomap[*,*,0]
    if keyword_set(minlat0) then begin
        floorlat = floor(minlat0)
        hilatmap = lat_deg gt floorlat
        spcmap = (spcmap ne 0)  and hilatmap
        dlat = 1-(minlat - floorlat)
        i = floorlat + 90
        spcmap[*,i] = spcmap[*,i] * dlat
    endif
    for i = 1, n_elements(unit) -1 do $
      spcmap = spcmap + unit[i] * geomap[*,*,i]
    
    ; hilatmap : high latitudes (eg not near south pol)
    ;   value = 1 if latitude > -31, 0 otherwise
    floorlat = floor(minlat)
    hilatmap = lat_deg gt floorlat

    ;spcmap = geomap[*,*,2] + geomap[*,*,5]
    ;
    ; geolmap : Grundy & Young 2004 N2 map - SPC and lat > -31
    geolmap = (spcmap ne 0)  and hilatmap

    ; take care of the fractional part
    dlat = 1-(minlat - floorlat)
    i = floorlat + 90
    geolmap = float(geolmap)
    geolmap[*,i] = geolmap[*,i] * dlat

    return, geolmap

end
