; NAME:
;  rdmap_gy04
; PURPOSE: (one line)
;  Read the Grundy Young 2004 N2 map of Triton
; CALLING SEQUENCE:
;  map_gy04 = rdmap_gy04(minlat, lat_deg, lon_deg)
; INPUTS:
;  sslat_deg - the subsolar latitude in degrees
; OUTPUTS:
;  map - 
;   360 by 180.  Pixels are 1-deg wide.
;   Pixel 0,0 runs from latitude -90 to -89 and East longitude 0 to 1.
;  lat_deg - the latitude in degrees (360 x 180 array)
;  lon_deg - the East longitude in degrees (360 x 180 )
; COMMON BLOCKS:
;  None
; RESTRICTIONS:
;  sslat must be -45 or -40
; MODIFICATION HISTORY:
;  Written 2007 June 26, by Leslie Young, SwRI
;-

function rdmap_gy04, minlat, lat_deg, lon_deg


; maps are all 360 x 180, with edge of the 0,0 pixel
; at 0 East longitude, -90 latitude

    physconstants               ; set up !phys.sigma, etc
    deg = !dpi/180.d
    
    ; read in the geologic map
    geomap = rdmap_triton_mcewen90(lat_deg, lon_deg)
                                ; geomap : 7 planes of geologic units
                                ;   0 = unmapped, 1-6 = units 1-6
                                ;   value = 0 if not in unit, 1 if in unit

    latmap = lat_deg * deg
    lonmap = lon_deg * deg

    ; spcmap : south polar cap (units 2+3+5)
    ;   value = 0 if not in units, 1 if in units
    spcmap = geomap[*,*,5] + geomap[*,*,2] + geomap[*,*,3]

    ; spc2map : south polar cap plus 
    ;   value = 0 if not in units, 1 if in units
    spc2map = geomap[*,*,5] + geomap[*,*,2] + geomap[*,*,3] + geomap[*,*,6]

    ; hilatmap : high latitudes (eg not near south pol)
    ;   value = 1 if latitude > -31, 0 otherwise
    floorlat = floor(minlat)
    hilatmap = lat_deg gt floorlat

    ; gy04map : Grundy & Young 2004 N2 map - SPC and lat > -31
    gy04map = spcmap and hilatmap

    ; take care of the fractional part
    dlat = 1-(minlat - floorlat)
    i = floorlat + 90
    gy04map = float(gy04map)
    gy04map[*,i] = gy04map[*,i] * dlat

    return, gy04map

end
