;+
; NAME:
;  triton_subsollat_2
; PURPOSE: (one line)
;  Return sub-solar latitude
; CALLING SEQUENCE:
;  slat = triton_subsollat_2(year, forget=forget)
; INPUTS:
;  year - year (e.g. 2000.)
; KEYWORD INPUT PARAMETERS:
;  forget - if set, use formula from Forget et al. 2000
; OUTPUTS:
;  slat - sub solar latitude (radian)
; PROCEDURE:
; MODIFICATION HISTORY:
;  2008 Mar 26
;-

function triton_subsollat_2, t, forget = forget

    if keyword_set(forget) then begin
        nf = n_elements(t)
        indxf = lindgen(nf)
        ns = 0
    endif else begin
        etmin = utc2et('1980 JAN 01 00:00:00.000')
        etmax = utc2et('2048 DEC 30 00:00:00.000')
        et = utcy2et(t)
        indxs = where(et gt etmin and et lt etmax, ns, $
                      comp = indxf, ncomp = nf)
    endelse
    
    slat = t ; to get the right size

    if nf gt 0 then begin

        tf = t[indxf]

        a = 0.429870
        b = 0.370543
        c = 0.0225091
        alpha0 = 61.52577 * !pi/180.
        alpha1 = 2.185376 * !pi/180.
        beta0 = -100.79473 * !pi/180.
        beta1 = 0.523169 * !pi/180.
        
        sinphi = a*cos(alpha0-alpha1*tf) + $
          b*sin(alpha0-beta0+(beta1-alpha1)*tf) + $
          c*sin(alpha0-beta0+(beta1+alpha1)*tf)
        slat[indxf] = - asin(sinphi)
    endif

    if ns gt 0 then begin
        ets = et[indxs]
        phys = naif_phys(801, ets)
        slat[indxs] = phys.subsollat
    endif

    return, slat

end
