; READ
; irtf_sites_v4.txt

root = 'irtf_sites_v2'

;===================================================
; STEP 1: READ IN THE TEXT FILE
;===================================================
infile = root+'.txt'

readcol, infile, idstr, mpcstr, allsites, nameshort, namelong, $
  nlatstr, wlonstr, altstr, datum, source, $ 
  for='(A,A,A,A,A, A,A,A,A,A)', $
  delim = '|', skip = 2
  
allsites = strtrim(allsites,2)
nameshort = strtrim(nameshort,2)
namelong = strtrim(namelong,2)

k = where(strmid(wlonstr,3,1) ne ':')
wlonstr[k] = ' ' + wlonstr[k]

k = where(float(altstr) lt 1000, nk)
if nk gt 0 then altstr[k] = ' ' + altstr[k]


nall = n_elements(nameshort)

;===================================================
; STEP 2: CREATE THE TK FILE (INPUT TO PINPOINT, USED FOR DOC, NAME)
;===================================================

; OUTPUT to <root>.tk

outfile = root + '.tk'
spkfile = root + '.spk'
pinpoint_cmd = 'pinpoint -def '+outfile+' -spk '+ spkfile + ' -pck pck00008.tpc'

openw, lun, outfile, /get_lun

;===================================================
; STEP 3: DOCUMENTATION TO FILE
;===================================================

; Start with initial documentation
printf,lun,'Created ',systime()
cd,CURR=cwd
printf,lun,cwd+'/'+outfile
printf,lun,'Observatory locations for Pluto2007Mar18 event from ' + infile

printf,lun,''
printf,lun,'What is in this file:'
printf,lun,'---------------------'
printf,lun,''
printf,lun,'Human-readable tables of sites and locations, with explanation'
printf,lun,'Definitions of name-code pairs'
printf,lun,'Input for pinpoint to create spk files'
printf,lun,''
printf,lun,'How to use this file'
printf,lun,'--------------------'
printf,lun,''
printf,lun,'With this file ('+outfile+') loaded, one can look up names and ids.'
printf,lun,'For example (in IDL)'
printf,lun,'  IDL> cspice_bodc2n,39990010,n,f'
printf,lun,'  IDL> help, n, f'
printf,lun,"  N               STRING    = 'RED-BUTTES'"
printf,lun,'  F               BYTE      =    1
printf,lun,"  IDL> cspice_bodn2c,'RBO',c,f"
printf,lun,'  IDL> help, c, f'
printf,lun,'  C               LONG      =     39990010'
printf,lun,'  F               BYTE      =    1'
printf,lun,''
printf,lun,'This file ('+outfile+') is used as the input to pinpoint with the command'
printf,lun,'  '+pinpoint_cmd
printf,lun,'to create '+spkfile
printf,lun,'pinpoint is available from http://naif.jpl.nasa.gov/naif/utilities.html'
printf,lun,''
printf,lun,'With '+spkfile+' loaded, one can calculate topocentric ephemerides'
printf,lun,'For example (in IDL)'
printf,lun,"  IDL> cspice_spkez,9,0,'J2000','LT',39990010,state,ltime"
printf,lun,'With both '+spkfile+'and '+outfile+' loaded, '$
  +'one can calculate topocentric ephemerides'
printf,lun,'For example (in IDL)'
printf,lun,"  IDL> cspice_spkezr,'Pluto',0,'J2000','LT','RBO',state,ltime"
printf,lun,''

printf,lun,'===========================================================

printf,lun,''
printf,lun, 'NAIF ID', $
  'NAME', 'SITE','ABBR', $
  'N_LAT', 'W_LON','ALT_M', 'DATUM',$
  'SOURCE'+'                                                              ',$
  form='(A8, " | ", '$
  + 'A53, " | ", A22, " | ", A8," | ",'$
  +'A12," | ", A13," | ", A7," | ", A6," | ", A25, " | ")'
for i=0, nall-1 do begin
    printf,lun, idstr[i], $
      namelong[i], allsites[i],nameshort[i], $
      nlatstr[i]+'    ', wlonstr[i]+'     ',altstr[i]+'   ', datum[i], source[i],$
  form='(A8, " | ", '$
  + 'A53, " | ", A22, " | ", A8," | ",'$
  +'A12," | ", A13," | ", A7," | ", A6," | ", A25, " | ")'
endfor

printf,lun,''
printf,lun,'About the NIAF IDs:'
printf,lun,'----------------'
printf,lun,''
printf,lun,'The IDs are 8 digits long.  They are long integers, the same'
printf,lun,'type returned by cspice_bodn2c.'
printf,lun,''
printf,lun,'The 8 digits are docomposed as follows:'
printf,lun,''
printf,lun,'Digits Meaning          Example'
printf,lun,'1 - 3  Body code.       399'
printf,lun,''
printf,lun,'       All sites so far start with 399 = Earth'
printf,lun,''
printf,lun,'4 - 7  Site code.       0568 (Mauna Kea)'
printf,lun,''
printf,lun,'       If the site has a 3-character code in the Minor Planet Circular'
printf,lun,'       list of sites, that code is used here.  Codes that begin with'
printf,lun,'       a digit (such as 662 Lick Observatory, Mount Hamilton) simply have'
printf,lun,'       a zero appended (so 662 -> 0662).  Sites that begin with a letter,'
printf,lun,'       (G96 Mt. Lemmon Survey) have the letter turned into a 2-digit'
printf,lun,'       number, with A -> 10, B -> 11, C -> 12, D -> 13, E -> 14, F -> 15,'
printf,lun,'       G -> 16, H -> 17, I -> 18, and J -> 19 (so G96 -> 1696).  '
printf,lun,''
printf,lun,'       If the site does not have a 3-character code in the Minor Planet Circular'
printf,lun,'       list of sites, we assign it a unique number beginning with the'
printf,lun,'       initial digit 9.'
printf,lun,''
printf,lun,'8      Telescope code.  '
printf,lun,''
printf,lun,'       If the site is in the Minor Planet Circular'
printf,lun,'       list, and we use that entry without modification, the'
printf,lun,'       telescope code is a zero.'
printf,lun,''
printf,lun,'       If the site is in the Minor Planet Circular, but we use'
printf,lun,'       a different position (because we have an'
printf,lun,'       independent measurement), this is non-zero.  Multiple'
printf,lun,'       coordinates, for more than one telescope at a site, are'
printf,lun,'       indicated by a telescope code of 1, 2, etc.'
printf,lun,'       (even if all telescopes use the Minor Planet Circular location)'
printf,lun,''

printf,lun,''
printf,lun,'About the names:'
printf,lun,'----------------'
printf,lun,''
printf,lun,'NAME: Long descriptive name'
printf,lun,''
printf,lun,'If the site is in the Minor Planet Circular list of sites,'
printf,lun,'the long descriptive name is taken from the "Name" column in that list. '
printf,lun,'Otherwise we devise a similarly descriptive name for the site.'
printf,lun,'If we are using multiple telescopes at one site, the telescope name is'
printf,lun,'appended in brackets.'
printf,lun,''
printf,lun,'SITE: shorter descriptive name'
printf,lun,''
printf,lun,'A shorter descriptive name.  In the case of multiple telescopes'
printf,lun,'at one site that are described by aperture, the aperture is'
printf,lun,'appended (as 10 * the apeture in meter).'
printf,lun,'These names are formatted so they can be used in variable names in the'
printf,lun,'pinpoint-input section (e.g., GRASSLANDS_IDCODE = 39906511)'
printf,lun,''
printf,lun,'ABBR: abbreviation for use in project plots and tables.'
printf,lun,''

printf,lun,''
printf,lun,'About the locations:'
printf,lun,'--------------------'
printf,lun,''
printf,lun,'Latitude and longitude are listed to the precision supplied, with the'
printf,lun,'datum, here known.  Recent datums (ITRF88-ITRF00 and WGS84) are identical'
printf,lun,'at the 1-m level.  IAU76 (used by Horizons to derive lat and lon from the'
printf,lun,'MPC tables) uses an equitorial radius of 6378140 m and a flattening of'
printf,lun,'1/298.257, while WGS84 uses 6378137 m and 1/298.257223563.   The difference'
printf,lun,'in location is less than a m horizontally, and 3 m vertically.  For this'
printf,lun,'project, we treat ITRDyy, IAU76, NAD83, and WGS84 identically.  '
printf,lun,''
;printf,lun,'The coordinate from GNTO were provided in NAD27, roughly 123 m different'
;printf,lun,'from NAD83, and were converted using tools at'
;printf,lun,'http://www.ngs.noaa.gov/TOOLS/Nadcon/Nadcon.shtml'

;===================================================
; STEP 4: MAKE THE NAME-CODE PAIRS
;===================================================

printf,lun,' '
printf,lun, '\begintext'
printf,lun,'===========================================================
printf,lun, 'The next section (after the \begindata) is used to
printf,lun, 'define name-code pairs, by abbreviated site name'
printf,lun, 'for use with bodn2c and bodc2n'
printf,lun,'===========================================================
unq = uniq(nameshort, sort(idstr))
nsites = n_elements(unq)
printf, lun, '\begindata'
for n=0, nsites-1 do begin
    i = unq[n]
    printf, lun, 'NAIF_BODY_CODE  += ( ', $
      idstr[i], $
      ' ) '
    printf, lun, 'NAIF_BODY_NAME  += ( ', $
      "'" + nameshort[i] + "'",  $
      ' ) '
end

printf,lun,' '
printf,lun, '\begintext'
printf,lun,'===========================================================
printf,lun, 'The next section (after the \begindata) is used to
printf,lun, 'define name-code pairs, by long name'
printf,lun, 'for use with bodn2c and bodc2n'
printf,lun,'===========================================================
unq = uniq(namelong, sort(idstr))
nsites = n_elements(unq)
printf, lun, '\begindata'
for n=0, nsites-1 do begin
    i = unq[n]
    printf, lun, 'NAIF_BODY_CODE  += ( ', $
      idstr[i], $
      ' ) '
    printf, lun, 'NAIF_BODY_NAME  += ( ', $
      "'" + namelong[i] + "'",  $
      ' ) '
end

printf,lun,''
printf, lun, '\begintext'

printf,lun,'===========================================================
printf,lun, 'The next section (after the \begindata) is used to
printf,lun, 'define name-code pairs, by site name'
printf,lun, 'for use with bodn2c and bodc2n'
printf,lun,'===========================================================
unq = uniq(allsites, sort(idstr))
printf, lun, '\begindata'
; First all the unique sites
for n=0, nsites-1 do begin
    i = unq[n]
    printf, lun, 'NAIF_BODY_CODE  += ( ', $
      idstr[i], $
      ' ) '
    printf, lun, 'NAIF_BODY_NAME  += ( ', $
      "'" + allsites[i] + "'",  $
      ' ) '
end


printf,lun,''
printf, lun, '\begintext'

;      \begindata
; 
;      NAIF_BODY_CODE  += ( 170100, 170101 )
; 
;      NAIF_BODY_NAME  += ( 'Enterprise', 'Enterprise-A' )


;===================================================
; STEP 3: CREATE THE INPUT TO PINPOINT
;===================================================

; get the unique ids
unq = uniq(idstr, sort(idstr))
nsites = n_elements(unq)

; convert to the format needed for the defs file
sites = allsites(unq)
siteIDs = long(idstr[unq])
FRAME="'ITRF93'"
lat_deg = decparse(nlatstr[unq]) * 180.d/!dpi
lon_deg = 360.d - decparse(wlonstr[unq]) * 180.d/!dpi
lon_deg = lon_deg mod 360.d
alt_km = double(altstr[unq])/1d3

; Write the defs file
printf,lun,''
printf,lun,'===========================================================
printf,lun, 'The next section (after the \begindata) is used by
printf,lun, 'pinpoint to create spk files for the above observatories
printf,lun,'===========================================================
printf,lun,'\begindata'
printf,lun
nsites=n_elements(sites)
printf,lun,"SITES =  ("
for n=0,nsites-1 do printf,lun,"      '"+sites[n]+"'"
printf,lun,")"
for n=0,nsites-1 do begin
	site=sites[n]
	printf,lun
	printf,lun,site+'_FRAME  = '+FRAME
	sIDcode=string(siteIDs[n],format='(I8)')
	printf,lun,site+'_IDCODE = '+sIDcode
	slatlonalt = string(lat_deg[n],lon_deg[n],alt_km[n],$
			    format='(F12.8,2x,F12.8,2x,F7.3)')
	printf,lun,site+'_LATLON = ('+slatlonalt+')'
	printf,lun,site+'_CENTER = 399'
endfor

printf,lun,''
printf, lun, '\begintext'

close, lun & free_lun, lun 

;===================================================
; STEP 3: MAKE THE SPK FILE
;===================================================

if isfile(spkfile) then file_delete, spkfile
print,pinpoint_cmd
spawn,pinpoint_cmd

end
