;+
;   Purpose of this routine is to compute extinction curves, from 2 or
;   more (hopefully lots more) solar analog star spectra.  Everything is
;   based on the contents of table 'night.extinction.tbl' (where night is
;   a string of the form '010121').  That table has lines which describe
;   two groups - first the group of spectra to be used in computing
;   the extinction, then the group of spectra to be corrected.  You
;   could generate this file by using 'night.groups.bat' and sticking
;   lists of star spectra at the beginning of each non-star line, and
;   appending a unique group name to the end of each of those lines.
;   (any line not having at least 3 entities is ignored).  Here's an
;   example:
;	1-4,12-19,80-87,260-263,276-279   56-63   Europa.01
;
;   All the spectra in each line of the file get corrected to SecZ = 0,
;   based on the extinction curve computed from the stars.  All of the
;   corrected object spectra get put in the directory "XXXXXX.corr"
;   (where XXXXXX represents the night code; this directory
;   must already exist).  Append a "noext" to the right of the group
;   name to prevent extinction from being calculated (useful if you
;   have just one star observation closely matched to the object's
;   airmass).  Alternatively "bracket" causes objects to be divided
;   by bracketing stars.  This is useful when the observer repeatedly
;   alternates between an object and a nearby star, but does not cover
;   a large range of airmasses.  This counts on stars being listed
;   sequentially.
;
;   Stars which are mediocre solar analogs can be corrected to T=5770K
;   by specifying their names (as they appear in night.tbl) and their
;   temperatures (in K), using the planck_star and planck_T keywords.
;
; 2004 Feb/March modification by Leslie Young (SwRI)
; - Signiticant reorganizing to improve readability of the
;   main routine (which previously had been 5 pages long itself!)
; - propogation of errors and keeping track of bad pixels
; - night.extinction.tbl - now takes 6 elements
;  skylist  objlist  objname  exttype  rawerrtype combtype
;  exttype = one of ['cfitexp','rfitexp','noext','bracket']
;    dictates how extinction is calculated and applied
;    cfitexp (default) - curvefit all stars to an exponential
;    rfitexp - robofit all stars to an exponential
;    noext - do no extinction correction
;    bracket - divide groups of objects by the mean 
;           of bracketing star group
;  rawerrtype = one of ['phot', 'gaus', 'medi']
;    how errors on raw spectra are calculated
;    phot - photon statistics (default)
;    gaus - deviation of points from line+gaussian
;    medi - median averaging
;  combtype = one of ['robo', 'wgt']
;    how spectra are combined
;    robo (default) - robomean
;    wgt - weighted mean
; - write out the following files
;   <night>.corr/corr.<ext>_<rawerr>.<num>.sp  the corrected spectrum
;   <night>.err/corr.<ext>_<rawerr>.<num>.sp   its errors, and
;   <night>.bad/corr.<ext>_<rawerr>.<num>.sp   its bad mask
;
;   <night>.final/<group>.<ext>_<rawerr>_<comb>.sp  the combined spectrum
;   <night>.err/<group>.<ext>_<rawerr>_<comb>.sp    its errors
;   <night>.bad/<group>.<ext>_<rawerr>_<comb>.sp    and bad mask
;   <night>.final/<group>.star.<ext>_<rawerr>_<comb>.sp  the superstar
;   These files are all written at the end of the spexred_extinction
;   routine, if we want to change the file names.
;
;-

; ==================================================================
; 
;    spexred_minme
;
; ==================================================================
PRO min_me, X, A, F, pder
    bx = exp(X*A[1])
    F = A[0]*bx
    IF n_params() ge 4 THEN pder = [[bx],[A[0]*X*bx]]
END

; ==================================================================
; 
;    spexred_tbl
;
; ==================================================================
PRO spexred_tbl, night, frame_num,beam,flag,object_name,secz, $
	date,timestr,time
    ; Read entries from night.tbl
    readcol,night+'.tbl',frame_num,beam,flag,object_name,secz, $
	date,timestr,format='(i,a,a,a,f,a,a)',/silent
    time = double(bytarr(n_elements(timestr)))
    ; convert time strings to decimal hours
    FOR i=0,n_elements(timestr)-1 DO BEGIN
	hms = strsplit(timestr[i],':',/extract)
	time[i] = double(hms[0]) + double(hms[1])/60.0 + double(hms[2])/3600.
    ENDFOR
END

; ==================================================================
; 
;    spexred_exttbl
;
; ==================================================================
PRO spexred_exttbl, night, stars, objects, gnames, exttype,rawerrtype,combtype
    ; Read groups from *.extinction.tbl
    nlines  = NUMLINES(night+'.extinction.tbl')
    ; Set up placeholders for arrays
    stars   = ['']
    objects = ['']
    gnames  = ['']
    exttype = ['']
    rawerrtype = ['']
    combtype = ['']
    ; Loop through the lines in *.extinction.tbl, looking for valid entries
    openr,lun,night+'.extinction.tbl',/get_lun
    FOR i=0,nlines-1 DO BEGIN
	temp = ' '	; make this variable be a string type
	readf,lun,temp
	temp = strtrim(temp,1)	;Remove any leading spaces
	; Use whitespace to divide string into separate items
	temp = strsplit(temp,' 	',/extract)
	; Ignore any comments or lines having less than 3 items
	IF strmatch(temp[0],'[0123456789]*') eq 1 AND $
	  n_elements(temp) ge 3 THEN BEGIN
	    ; Ok, this one must be a good line, record the info
	    stars   = [stars,temp[0]]
	    objects = [objects,temp[1]]
	    gnames  = [gnames,temp[2]]
            exttypestr = ' '
            rawerrtypestr = ' '
            combtypestr = ' '
	    ; Check if there's any other flags (like "noext") set
	    IF n_elements(temp) ge 4 THEN BEGIN
                exttypestr = strlowcase( temp[3] )
                IF n_elements(temp) ge 5 THEN BEGIN
                    rawerrtypestr = strlowcase( temp[4] )
                    IF n_elements(temp) ge 6 THEN BEGIN
                        combtypestr = strlowcase( temp[5] )
                    ENDIF
                ENDIF
	    ENDIF
            exttypestr = spexred_default(exttypestr, $
                                   ['cfitexp','rfitexp','noext','bracket'])
            rawerrtypestr = spexred_default(rawerrtypestr, $
                                          ['phot','gaus','medi'])
            combtypestr = spexred_default(combtypestr, $
                                          ['robo','wgt'])
            exttype = [exttype, exttypestr]
            rawerrtype = [rawerrtype, rawerrtypestr]
            combtype = [combtype, combtypestr]
        ENDIF
    ENDFOR
    close,lun
    ; Keep all but the initial placeholder
    z = indgen(n_elements(stars)-1)+1
    stars   = stars[z]
    objects = objects[z]
    gnames  = gnames[z]
    exttype   = exttype[z]
    rawerrtype   = rawerrtype[z]
    combtype   = combtype[z]
END

; ==================================================================
; 
;    spexred_indx
;
; ==================================================================
; return indices into night.tbl arrays
FUNCTION spexred_indx, indx, frame_num
    ni = n_elements(indx)
    frame_indx = intarr(ni)
    for i=0,ni-1 do begin
        frame_indx[i]= ( where(indx[i] eq frame_num) )[0]
    end
    return, frame_indx
END


; ==================================================================
; 
;    spexred_rangepar
;
; ==================================================================
; expand star or object lists into indices, excluding stars
; marked 'x' in night.tbl.  Written with variable names 
; originally used for the star lists, but works equally well
; for object lists 
PRO spexred_rangepar, stars, frame_num, flag, strs, ns

	; Parse the ranges of file names
	rangepar,stars,strs

	; Exclude any spectra that are marked "x" in night.tbl
        z = where( flag[spexred_indx(strs,frame_num)] ne 'x' , count)
        if count eq 0 then begin
            strs = [-1]
            ns = -1
            return
        end
	strs = strs[z]
	ns = n_elements(strs)
  
END

; ==================================================================
; 
;    spexred_rdgrp
;
; ==================================================================
; Read in a group of stars or objects
; if flag star is set then the routine corrects for planck stars
; and sets peg_o, peg_z1, peg_z2.  Otherwise, uses peg_* to 
; normalize the object spectra
PRO spexred_rdgrp, night,norder,frame_num,object_name,secz, $
                   wavl,strs, rawerrtype, $
                   array,earray,barray,seczs, peg_o,peg_z1,peg_z2, $
                   planck_star=planck_star,planck_T=planck_T, $
                   star=star

        ; Read in all the star data we'll need
        ; for the extinction curve
        ns = n_elements(strs)
	array  = double(bytarr(ns,norder,1024))
	earray  = double(bytarr(ns,norder,1024))
	barray = bytarr(ns,norder,1024)
	seczs  = double(bytarr(ns))

	; Coarsly bin the data so we can figure out where highest SNR is
	nbin = 20
	temp_bins_s = fltarr(ns,norder,nbin)
	temp_bins_n = fltarr(ns,norder,nbin)
	temp_bins_z1 = intarr(nbin)
	temp_bins_z2 = intarr(nbin)
	; Figure out starting and stopping pixels for each bin
	FOR j=0,nbin-1 DO BEGIN
	    temp_bins_z1[j] = (1024/nbin)*j
	    temp_bins_z2[j] = (1024/nbin)*(j+1) - 1
        ENDFOR

	; Loop through all the star spectra
	FOR i=0,ns-1 DO BEGIN
	    ; Read in this spectrum and save it in array[i,*,*]
	    num = string(format='(i4.4)',strs[i])
;	    data = readfits(night+'.raw/raw'+num+'.sp',/silent)
            err = spexred_rawerror(night,num,data,rawerrtype=rawerrtype)
	    bad = readfits(night+'.bad/bad'+num+'.sp',/silent)
            z=where(err eq -100, count)
            if count ne 0 then bad[z] = 1

	    ; Correct star to solar temperature 5770 K if necessary
	    IF keyword_set(planck_star) and keyword_set(star) THEN BEGIN
		a = where(frame_num eq strs[i])
		temp_on = object_name[a]
		; This convoluted test is required because object_name[a]
		; wants to be an array, rather than a simple string
		z = where(planck_star eq $
		    string(object_name[a],format='(a)'),count)
		IF count eq 1 THEN BEGIN
		    FOR o=0,norder-1 DO BEGIN
			; reforms and total are included to ensure
			; correct dimensions of planckfun() result.
			norm = planckfun(reform(wavl[o,*])*.0001, $
			    total(planck_T[z]))/ $
			    planckfun(reform(wavl[o,*])*.0001,5770.)
			data[o,*] = data[o,*]*norm
			err[o,*] = err[o,*]*norm
                    ENDFOR
                    z = where(bad eq 1, count)
                    IF count gt 0 then err[z] = -100.
		ENDIF
            ENDIF

	    array[i,*,*] = data
	    earray[i,*,*] = err
	    barray[i,*,*] = bad
	    ; We need a spot where we can assume zero extinction for
	    ; normalization purposes.  Accumulate data in temp_bins_*
	    ; so we can pick a nice, high SNR spot for this purpose.
	    FOR o=0,norder-1 DO BEGIN
		FOR j=0,nbin-1 DO BEGIN
		    x = data[o,temp_bins_z1[j]:temp_bins_z2[j]]
		    b = bad[o,temp_bins_z1[j]:temp_bins_z2[j]]
		    z = where(b eq 0,count)
		    IF count gt 2 THEN BEGIN
			robomean,x[z],4.0,.1,avg,avgdev,stddev
			temp_bins_s[i,o,j] = avg
			temp_bins_n[i,o,j] = stddev
		    ENDIF ELSE BEGIN
			temp_bins_s[i,o,j] = 0.0
			temp_bins_n[i,o,j] = 1.0
		    ENDELSE
		ENDFOR
	    ENDFOR
	    ; Record the airmass of this observation
	    seczs = secz[spexred_indx(strs,frame_num)]
	ENDFOR


	; Figure out how to normalize all the star spectra
        IF keyword_set(star) THEN BEGIN
            min_snrs = fltarr(norder*nbin)
            ; Look for the least bad wavelength, in terms of minimum SNR
            FOR o=0,norder-1 DO BEGIN
                FOR b=0,nbin-1 DO min_snrs[o*nbin + b] = $
                  min(temp_bins_s[*,o,b]^2/temp_bins_n[*,o,b])
            ENDFOR
            peg = where(min_snrs eq max(min_snrs))
            ; peg = o*nbin + b is the spot in temp_bins_* we want
            peg_b = peg mod nbin
            peg_o = (peg - peg_b)/nbin
            ; figure out the pixel limits of the peg spot
            peg_z1 = (1024/nbin)*peg_b
            peg_z2 = (1024/nbin)*(peg_b+1) - 1
        ENDIF

	FOR i=0,ns-1 DO BEGIN
	    ; Normalize each star to the median in the peg spot
	    x = array[i,peg_o,peg_z1:peg_z2]
	    b = barray[i,peg_o,peg_z1:peg_z2]
	    norm = 1.0/median(x[where(b eq 0)])
	    array[i,*,*] = array[i,*,*]*norm
	    earray[i,*,*] = earray[i,*,*]*norm
        ENDFOR
        z = where(barray eq 1, count)
        if count gt 0 then earray[z] = -100.

END

; ==================================================================
; 
;    spexred_comb
;
; ==================================================================
; combine an array of spectra according to method combtype,
; returning the errors from scatter
; optionally returning the "internal error" (from propagation)
PRO spexred_comb, array, earray, barray, super, supererr, superbad, $
                  combtype, superinterr=superinterr 

sz = size(array)
ns = sz[1]
norder = sz[2]

super = fltarr(norder,1024) + 1.0
supererr = fltarr(norder,1024) - 100.
superinterr = fltarr(norder,1024) - 100.
superbad = bytarr(norder,1024) + 1

FOR o=0,norder-1 DO BEGIN
    FOR k=0,1023 DO BEGIN
        z = where(reform(barray[*,o,k]) eq 0,count) 
        IF count eq 0 THEN BEGIN
            CONTINUE
        ENDIF

        x = array[z,o,k] 
        e = earray[z,o,k] 
        IF count eq 1 THEN BEGIN
            super[o,k] = x[0]
            superinterr[o,k] = e[0]
            CONTINUE
        ENDIF

        CASE combtype OF
            'robo': BEGIN  ;  Robust mean ----------------------------
                IF count gt 5 THEN BEGIN
                    robomean,x,3.5,0.1,avg, $
                      avgdev,stddev,var,skew,kurt,nfinal
                    IF nfinal GT 1 THEN BEGIN
                        super[o,k] = avg
                        supererr[o,k] = sqrt(var/nfinal)
                        superinterr[o,k] = sqrt(total(e^2))/count
                        superbad[o,k] = 0 
                    ENDIF
                ENDIF ELSE BEGIN
                    avg = mean(x)
                    super[o,k] = avg
                    var=total( (x-avg)^2 ) /(count-1)
                    supererr[o,k] = sqrt(var/count)
                    superinterr[o,k] = sqrt(total(e^2))/count
                    superbad[o,k] = 0 
                ENDELSE
            END
            
            'wgt': BEGIN   ;  Robust mean ----------------------------
                w = 1/e^2. 
                wtot = total(w)
                avg = total(x*w)/wtot
                super[o,k] = avg
                superinterr[o,k] = sqrt(1/wtot)
                chisq = total(w * (x-avg)^2/(count-1))
                supererr[o,k] = sqrt(chisq/wtot) 
                superbad[o,k] = 0 
            END
        ENDCASE
    ENDFOR
ENDFOR
END

; ==================================================================
; 
;    spexred_extinction
;
; ==================================================================
PRO spexred_extinction,night,extinc=extinc,airmass=airmass, $
	planck_star=planck_star,planck_T=planck_T

    IF keyword_set(planck_star) THEN BEGIN
	wavl = readfits('wavl.fits',/silent)
	IF not keyword_set(planck_T) THEN BEGIN
	    print,'ERROR: no temperature specified with planck_T keyword'
	    return
	ENDIF
    ENDIF ELSE BEGIN
	IF keyword_set(planck_T) THEN BEGIN
	    print,'ERROR: no star specified with planck_star keyword'
	    return
	ENDIF
    ENDELSE

    ; Read in information about how many orders to use and where they're OK
    norder = spexred_norder(keep=keep)

    ; Read table data for the night
    spexred_tbl, night, frame_num,beam,flag,object_name,secz, $
	date,timestr,time

    ; Read groups from *.extinction.tbl
    spexred_exttbl, night, stars, objects, gnames, $
      exttype, rawerrtype, combtype
    noext   = exttype eq 'noext' or exttype eq 'bracket' 
    bracket = exttype eq 'bracket'

    nl = n_elements(stars)
    ; Loop through the lines in night.extinction.tbl
    FOR l=0,nl-1 DO BEGIN

	; Parse the ranges of file names excluding 'x'ed spectra & 
        ; Read in all the star data we'll need for the extinction curve
        ; and set peg_o, peg_z1, peg_z2
	spexred_rangepar, stars[l], frame_num, flag, strs, ns
        spexred_rdgrp, night,norder,$
                   frame_num,object_name,secz,wavl,strs, rawerrtype[l], $
                   star_array,star_earray,star_barray,star_seczs, $
                   peg_o,peg_z1,peg_z2, $
                   planck_star=planck_star,planck_T=planck_T, $
                   /star

        ; Parse and read the object ranges, using (not setting) peg_*
	spexred_rangepar, objects[l], frame_num, flag, objs, no
        spexred_rdgrp, night,norder,$
          frame_num,object_name,secz,wavl,objs, rawerrtype[l], $
          obj_array,obj_earray, obj_barray,obj_seczs, $
          peg_o,peg_z1,peg_z2

	; Compute the extinction curve from all the star spectra
        ; This is really awkward.  I'd much rather see a big
        ; CASE statement that calculated the extinction
        ; and corrected the obj_arrays depending on the
        ; value of exttype
	extinc = fltarr(norder,1024)
	FOR i=0,norder-1 DO BEGIN
	    FOR k=0,1023 DO BEGIN
		z = where(reform(star_barray[*,i,k]) eq 0,count)
		IF count lt 2 OR noext[l] eq 1 THEN BEGIN
		    ; Insufficient data in this region,
		    ; or else no extinction correction is wanted.
		    extinc[i,k] = 0.0
		ENDIF ELSE BEGIN
		    y = reform(star_array[z,i,k])
		    x = star_seczs[z]
                    result = [mean(y),-.05]
                    e = x*0.0 + 1.0 ; equal weights for all points
                    if exttype[l] eq 'cfitexp' then begin
                        yfit = curvefit(x,y,e,result, $
                                        FUNCTION_NAME='min_me',$
                                        /DOUBLE,itmax=100, iter=iter)
                    endif else begin  ; exttype eq 'rfitexp'
                        yfit = robofit(x,y,e,result, $
                                       FUNCTION_NAME='min_me',$
                                       itmax=100,iter=iter,thresh=3.)
                    end
                    IF iter lt 100 THEN BEGIN
                        ; Only keep the exponential part, not the scale factor
                        extinc[i,k] = result[1]
                    END
		ENDELSE
	    ENDFOR
	ENDFOR
	; Apply the extinction curve to all the stars (correct to 0 airmass)
	FOR s=0,ns-1 DO BEGIN
            fact = exp(-star_seczs[s]*extinc)
            star_array[s,*,*] = star_array[s,*,*]*fact
            star_earray[s,*,*] = star_earray[s,*,*]*fact
        END
        z = where(star_barray eq 1, count)
        IF count NE 0 THEN star_earray[z] = -100.
	airmass = star_seczs	; return star airmasses back to the user

	; -----------------------------------------------------------------
	IF bracket[l] eq 1 THEN BEGIN ; -----------------------------------
	    ; If bracketing is set, need to compute a series of superstars,
	    ; instead of just one.

	    ; Bogus superstar = a vector of ones, since we perform the
	    ; ratio here instead of at the end, as is done for the
	    ; non-"bracket" case.
	    superstar = fltarr(norder,1024) + 1.0

	    ; Break the stars and objects into groups, based on
	    ; location of commas, and further
	    ; bust up star_groups into integer maxs and mins
            star_groups = strsplit(stars[l],',',/extract)
	    str_max = lonarr(n_elements(star_groups))
	    str_min = lonarr(n_elements(star_groups))
	    FOR i=0,n_elements(star_groups)-1 DO BEGIN
		m = strsplit(star_groups[i],'-',/extract)
		str_min[i] = long(m[0])
		str_max[i] = long(m[1])
	    ENDFOR

            obj_groups = strsplit(objects[l],',',/extract)
	    ; Groups are contained as arrays of strings of the form 'nn-nn'
	    ; in obj_groups and star_groups.  Loop through object groups:
	    FOR i = 0,n_elements(obj_groups)-1 DO BEGIN
		; Break this object group into an array of object numbers
		rangepar,obj_groups[i],obj_gr
		str_gr = [-1]	; Placeholder
		; Figure out bracketing star group(s) and convert the
		; relevant stars to an array of star numbers: str_gr
		z = where(str_max lt min(obj_gr),count)
		IF count gt 0 THEN BEGIN
		    rangepar,star_groups[z[count-1]],temp_strs
		    str_gr = [str_gr,temp_strs]
		ENDIF
		z = where(str_min gt max(obj_gr),count)
		IF count gt 0 THEN BEGIN
		    rangepar,star_groups[z[0]],temp_strs
		    str_gr = [str_gr,temp_strs]
		ENDIF
		IF n_elements(str_gr) eq 1 THEN BEGIN
		    print,'ERROR: No corresponding stars for objects ' $
			+ obj_groups[i] +'.'
		    return
		ENDIF
		str_gr = str_gr[1:n_elements(str_gr)-1] ; strip placeholder out
		; s_list points to these stars in star_array and star_barray
		s_list = [-1]
		FOR s=0,n_elements(str_gr)-1 DO $
		    s_list = [s_list,where(strs eq str_gr[s])]
		s_list = s_list[where(s_list ne -1)]
		; o_list points to these objects in obj_array and obj_barray
		o_list = [-1]
		FOR o=0,n_elements(obj_gr)-1 DO $
		    o_list = [o_list,where(objs eq obj_gr[o])]
		o_list = o_list[where(o_list ne -1)]

		; Build superstar from this group of stars
		; (set to 1 where no data exist, to avoid division by zero)
                spexred_comb, star_array[s_list,*,*], $
                  star_earray[s_list,*,*], $
                  star_barray[s_list,*,*], this_superstar, $
                  this_superstarerr, this_superstarbad, combtype[l]

		; Loop through, dividing each member of this group of objects
		; by the superstar.  
		FOR o=0,n_elements(o_list)-1 DO BEGIN
		    num = string(format='(i4.4)',obj_gr[o])
                    f = 1/this_superstar
		    obj_array[o_list[o],*,*] = obj_array[o_list[o],*,*] * f
		    obj_earray[o_list[o],*,*] = obj_earray[o_list[o],*,*] * f
                ENDFOR
                z = where(obj_barray eq 1, count)
                IF count gt 0 THEN obj_earray[z] = -100.
		; Merge all the objects in this group
                spexred_comb, obj_array[s_list,*,*], obj_earray[s_list,*,*], $
                  obj_barray[s_list,*,*], this_superobj, this_superobjerr, $
                  this_superobjbad, combtype[l]
		; Normalize this superobject
                norm = 1/median(this_superobj[peg_o,peg_z1:peg_z2])
		this_superobj = this_superobj * norm
		this_superobjerr = this_superobjerr * norm
                z = where(this_superobjbad eq 1, count)
                IF count NE 0 THEN this_superobjerr[z] = -100.
		    
		; Save resulting superobj for this group
		m1 = strtrim(string(obj_gr[i],format='(i)'),1)
		writefits,night+'.final/'+gnames[l]+'.'+m1+'.1.sp', $
		    this_superobj
		writefits,night+'.final/'+gnames[l]+'.'+m1+'.star.1.sp', $
		    this_superstar
	    ENDFOR

	ENDIF ELSE BEGIN ; ------------------------------------------------
	; -----------------------------------------------------------------
	    ; Without bracketing, compute a single superstar and superobj
	    ; pair and do the ratio between those (at the end).

	    ; Compute the grand average SecZ=0 star spectrum "superstar"
            spexred_comb, star_array, star_earray, star_barray, $
              superstar, superstarerr, superstarbad, combtype[l]
            z = where(superstar eq 0., count)
            IF count GT 0 THEN BEGIN
                superstar[z] = 1.
                superstarerr[z] = -100
                superstarbad[z] = 1
            END

	    ; Next we correct the object spectra for extinction
            ; Why don't we go ahead and divide by the superstar
            ; here?  Then all methods have the air-mass and
            ; star-divided files in corr.  Otherwise, bracket
            ; has air- and star-, others have only air corrected.
	    FOR i=0,no-1 DO BEGIN
		; Apply airmass/extinction correction
                fact = exp(-obj_seczs[i]*extinc)
		obj_array[i,*,*] = obj_array[i,*,*]*fact
                obj_earray[i,*,*] = obj_earray[i,*,*]*fact
	    ENDFOR
            z = where(obj_barray eq 1, count)
            IF count GT 0 THEN obj_earray[z] = -100.

        ENDELSE
 ; -----------------------------------------------------------------

        ; write the corrected spectra
        FOR i=0,no-1 DO BEGIN
            num = string(format='(i4.4)',objs[i])
            fn = 'corr.' + exttype[l]+'_'+rawerrtype[l]+'.'+num+'.sp'
            writefits,night+'.corr/'+fn,reform(obj_array[i,*,*])
            writefits,night+'.err/'+fn,reform(obj_earray[i,*,*])
            writefits,night+'.bad/'+fn,reform(obj_barray[i,*,*])
        ENDFOR

	; Compute grand object average
        spexred_comb, obj_array, obj_earray, obj_barray, $
          superobj, superobjerr, superobjbad, combtype[l]
        z = where(superobj eq 0., count)
        IF count GT 0 THEN BEGIN
            superobj[z] = 1.
            superobjerr[z] = -100
            superobjbad[z] = 1
        END

        ; Calculate the final ratio 
        oferr = superobjerr/superobj
        sferr = superstarerr/superstar
        superobj = superobj/superstar
        superobjerr = superobj * sqrt(oferr^2 + sferr^2)
        ; Normalize the superobject
        norm = 1/median(superobj[peg_o,peg_z1:peg_z2])
        superobj = superobj * norm
        superobjerr = superobjerr * norm
        z = where(superobjbad eq 1, count)
        IF count NE 0 THEN superobjerr[z] = -100.

	; Save final ratio (and also the star, so we can get back to raw)
        fn = gnames[l]+'.'+ exttype[l]+'_'+rawerrtype[l]+'_'+combtype[l]+'.sp'
	writefits,night+'.final/'+fn,superobj
	writefits,night+'.err/'+fn,superobjerr
	writefits,night+'.bad/'+fn,superobjbad
        fn = gnames[l]+'.star.'+ $
          exttype[l]+'_'+rawerrtype[l]+'_'+combtype[l]+'.sp'
	writefits,night+'.final/'+fn,superstar
	; (in case of "bracket" the superstar is just a boring vector of ones).

	; Save group avg time and minmax times
	IF l gt 0 THEN append=1 ELSE append=0
        obj_times=time[spexred_indx(objs,frame_num) ]
        obj_time_mm = minmax(obj_times)
	openw,1,night+'.groups.times',append=append
	printf,1,gnames[l]+"                   ",mean(obj_times), $
	    obj_time_mm[0],obj_time_mm[1],format='(a20,d12.9,d12.8,d12.8)'
	close,1

    ENDFOR
END
