; To do
; add sky flux to photon-counting calculation (if I can figure out how!)
; take a method flag
; write the data to night.err/rawerrphot<num>,sp, rawerrmedi<num>.sp, etc
; OK to write the first time it's called, just return after that
; 

;+
; 
; NAME:
;  spexred_rawerror
; PURPOSE: (one line)
;  Calculate error bars for raw data
; DESCRIPTION:
;  Calculate error bars using (1) counting statistics and (2) relative scatter
; CATEGORY:
;  spectral analysis
; CALLING SEQUENCE:
;  err = spexred_rawerror(night,num,data,recalc=recalc, rawerrtype=rawerrtype)
; INPUTS:
;  night   - yymmdd string for the UT night
;  num     - spectrum number
; OPTIONAL INPUT PARAMETERS:
; INPUT KEYWORD PARAMETERS:
;  recalc - recalculate, even if file exists (default=no)
;  rawerrtype - switch for error estimate (default=phot) 
;    'phot' - photon counting statistic
;    'gaus' - robost comparison with a line+gaussian
;    'medi' - median averaging of data, followed by median averaging
;             of errors
; OUTPUT KEYWORD PARAMETERS:
;  None
; OUTPUTS:
;  data - array of raw data [5,1024]
;  error - array of error estimates (-100 for bad pixels)
; COMMON BLOCKS:
;  None.
; SIDE EFFECTS:
;  writes <night>err.rawrrr<type><num>.sp, err[norder,1024]
; RESTRICTIONS:
;  None.
; PROCEDURE:
; MODIFICATION HISTORY:
;  Written by Leslie A. Young, SwRI, 2004 Feb 19
;-

function spexred_rawerror,night,num,data,recalc=recalc, rawerrtype=rawerrtype

; set keywords
if not keyword_set(recalc) then recalc = 0
if not keyword_set (rawerrtype) then rawerrtype = 'phot'
rawerrtype = spexred_default(rawerrtype,['phot','gaus','medi'])

; read in the raw data
rawfn = night+'.raw/raw'+string(num,form='(i4.4)')+'.sp'
if not isfile(rawfn) then begin
    data = fltarr(norder,1024)
    error = data - 100.
    print, 'spexred_rawerror: raw data file '+rawfn+' not found'
    return, error
endif
data = readfits(rawfn,/silent)

; output variables
outdir=night+'.err/'
outfn=outdir+'raw'+rawerrtype+string(num,form='(i4.4)')+'.sp'

if not isfile(outdir) then file_mkdir, outdir
; if we've already calculated the errors with this method 
; and a recalculation is not forced, then just return old errors 
if isfile(outfn) and recalc eq 0 then begin
    error = readfits(outfn, errorh, /silent)
    return, error
end

; -------------------------------------------------------
; File's not present or recalc is forced. 

; Read in information about how many
; orders to use and where they're OK
norder = spexred_norder(keep=keep)
; make a bad mask out of the information
keep_mask = bytarr(n_elements(keep),1024) + 1
FOR i=0,n_elements(keep)-1 DO BEGIN &$
    cmd = 'keep_mask'+keep[i]+' = 0' &$
    tst = execute(cmd) &$
ENDFOR

; Find the outliers by comparing to a smooth version of the spectrum.  
; Some orders are only valid in the middle of the array 
; (as returned by spexred_norder). In this use the valid portion.
bad = bytarr(norder,1024) + 1
FOR i=0,norder-1 DO BEGIN
    gd = where(keep_mask[i,*] eq 0)
    mm = minmax(gd)
    bad[i,mm[0]:mm[1]]=spexred_outlier(data[i,mm[0]:mm[1]])
ENDFOR

case rawerrtype of

;---------------------- phot - photon counting -----------------------
'phot': begin
    gain = 13.                  ; electron/dn
    rn = 12.                    ; readnoise in electrons
    spatialwidth = 7.           ; approximate width of profile
    
    ; read in the sky - not yet used
    ; sky = readfits(night+'.sky/sky'+string(num,form='(i4.4)')+'.sp',/silent)
    
    ; calculate the propogated error
    error = sqrt(gain*(data>0) + rn^2*spatialwidth)/gain

    bd = where(bad, nbd)
    if nbd gt 0 then error[bd] = -100.
end

;---------------------- gaus - fit of gaussian+line-------------
'gaus': begin
    ; assume each point is "really" a + b * gaussian
    error = fltarr(5,1024)
    specwidth = 2.5 ; fwhm of gaussian
    w = specwidth/(2.*sqrt(alog(2.)))
    nh = ceil(2*w) 
    n=2*nh+1
    x=(findgen(n)-nh)/w
    func='kerlinefunct'
    ww = replicate(1.,n)
    for order = 0,4 do begin 
        for i=nh,1023-nh do begin ; middle of the order
            yy=reform(data[order,i-nh:i+nh])
            gd = where(bad[order,i-nh:i+nh] eq 0, ngd)
            if ngd lt 4 or bad[order,i] eq 1 then begin
                error[order,i] = -100.
            endif else begin
                a = [mean(yy),0.,0.]
                yfit=curvefit(x[gd],yy[gd],w[gd], a, sig, $
                              FUNC=func, CHISQ=chisq)
                error[order,i]=sqrt( chisq )/sqrt(n-1.)
            endelse
        endfor
        ee = error[order,nh:nh+n]  ; start of the order  
        gd=where( ee gt 0, ngd)
        if ngd eq 0 then begin
            error[order,0:nh-1] = -100.
        endif else begin
            error[order,0:nh-1] = mean(ee[gd])
        endelse
        ee = error[order,1023-nh-n:1023-nh] ; end of the orde
        gd=where( ee gt 0, ngd)
        if ngd eq 0 then begin
            error[order,1024-nh:1023] = -100.
        endif else begin
            error[order,1024-nh:1023] = mean(ee[gd])
        endelse
    endfor
end

;---------------------- medi - compare to gross median filtering ------
'medi': begin
    ; assume each point is "really" a + b * gaussian
    error = fltarr(5,1024)
    specwidth = 5 ; boxcar width for final error calc (should be odd)
    nh = floor((specwidth-1)/2.)
    for order = 0,4 do begin 
        for i=nh,1023-nh do begin
            if error[order,i] ne -100. then begin
                y=reform(data[order,i-nh:i+nh])
                gd=where(bad[order,i-nh:i+nh] ne 1,ngd)
                if ngd le 2 or bad[order,i] then begin
                    error[order,i] = -100.
                endif else begin
                    yy = y[gd]
                    ymed = median(yy,/even)
                    yadev = mean(abs(yy-ymed))
                    error[order,i] =yadev/0.8
                endelse
            end
        end
        ee = error[order,nh:nh+specwidth]  ; start of the order  
        gd=where( ee gt 0, ngd)
        if ngd eq 0 then begin
            error[order,0:nh-1] = -100.
        endif else begin
            error[order,0:nh-1] = mean(ee[gd])
        endelse
        ee = error[order,1023-nh-specwidth:1023-nh] ; end of the orde
        gd=where( ee gt 0, ngd)
        if ngd eq 0 then begin
            error[order,1024-nh:1023] = -100.
        endif else begin
            error[order,1024-nh:1023] = mean(ee[gd])
        endelse
    endfor
end

endcase ; finish calculating error

; write out the file
writefits, outfn, error

return, error

end


pro spexred_rawerrortest,num

window,5,xs=600,ys=800
multi=!p.multi
!p.multi=[0,0,4]
night='020715'
mn=842
mx=882
order=2
n=1+mx-mn
errphot = spexred_rawerror(night,num,raw,ra='phot', /rec)
errgaus = spexred_rawerror(night,num,raw,ra='gaus', /rec)
errmedi = spexred_rawerror(night,num,raw,ra='medi', /rec )
ploterr,findgen(n),raw[order,mn:mx],errphot[order,mn:mx],/xs,ps=4, tit='phot'
ploterr,findgen(n),raw[order,mn:mx],errgaus[order,mn:mx],/xs,ps=4, tit='gaus'
ploterr,findgen(n),raw[order,mn:mx],errmedi[order,mn:mx],/xs,ps=4, tit='medi'
plot, findgen(n), errgaus[order,mn:mx],/li,/xs
oplot, findgen(n), errphot[order,mn:mx]
oplot, findgen(n), errmedi[order,mn:mx], li=2

!p.multi=multi
end
