;+
;
;	Routine to read wavl calibration and a SpeX spectrum, and
;	display each spectral order in a distinct color.
;
;	User needs to specify either a fits file name or an array
;	configured like [order, pixel number].
;
; KEYWORD OUTPUTS:
;   data      - An array to recieve plotted data pairs arranged such
;		that data[0,*] is wavelength and data[1,*] is signal.
;
; KEYWORD INPUTS:
;   file      - Input data file (a FITS file having dimensions
;		[norders,npoints]) which is matched by the dimensions
;		of "wavl.fits".
;   array     - Alternative input from an IDL array (again, having
;               dimensions [norders,npoints]).  The FILE and ARRAY
;		input forms are mutually exclusive.
;   wavl      - An array of wavelengths (default is to just read
;		wavelengths from "wavl.fits").
;   title     - Put a title on the plot.
;   psym      - Specify a plotting symbol.
;   xr,yr     - Specify plot boundaries.
;   keep      - An array of strings, each containing an argument
;		describing an order to be plotted.  A typical one
;		of these could be '[0,*]' meaning use all the points
;		from order number zero.  To ignore part of an order,
;		do something like this: '[5,0:473]'.  Default behavior
;		is to use spexred_norder() to read keep from a file.
;   bad       - array of [norder,1024] with 0 for good, 1 for bad
;   err       - array of [norder,1024] error bars
;
; MODIFICATION HISTORY
;  2004 Mar 1  LAY SwRI added bad and err keywords to Will Grundy's code
;-

PRO spexred_show_spec, file=file, array=array, yr=yr, xr=xr, $
	wavl=wavl, $
	psym=psym, data=data, title=title, keep=keep, noerase=noerase, $
        bad=bad, err=err 

    IF (not keyword_set(file) AND not keyword_set(array)) $
	OR (keyword_set(file) AND keyword_set(array)) THEN BEGIN
	print,'ERROR: Need to specify file or array (but not both!)'
	return
    ENDIF
    IF not keyword_set(keep) THEN norder = spexred_norder(keep=keep)
    IF not keyword_set(title) THEN BEGIN
	IF keyword_set(file) THEN title = file ELSE title = ''
    ENDIF

    color = ['100','170','100','170','100','170','100','170','100']
    ; Get wavelengths calibration, either from input array or wavl.fits
    IF not keyword_set(wavl) THEN BEGIN
	; Get wavelengths from wavl.fits file
	st = findfile('wavl.fits',count=count)
	IF count eq 0 THEN BEGIN
	    print,'ERROR: could not find "wavl.fits" file.'
	    return
	ENDIF
	wavl = readfits('wavl.fits')
    ENDIF

    IF keyword_set(file) THEN data = readfits(file)
    IF keyword_set(array) THEN data = array

    IF not keyword_set(yr) THEN yr = [0,1.3]
    IF not keyword_set(xr) THEN BEGIN
	cmd = 'xr1 = minmax(wavl'+keep[0]+')'
	tst = execute(cmd)
	cmd = 'xr2 = minmax(wavl'+keep[n_elements(keep)-1]+')'
	tst = execute(cmd)
	xr = minmax([xr1,xr2])
    ENDIF

    IF not keyword_set(bad) THEN bad = bytarr(norder,1024)

    plot,[0],[0],yr=yr,xr=xr,xtit='Wavelength (microns)', $
	xsty=1,ysty=1,tit=title,noerase=noerase
    spec_w = [-999.99]
    spec_d = [-999.99]
    keep_mask = bytarr(n_elements(keep),1024) 
    FOR i=0,n_elements(keep)-1 DO BEGIN
        cmd = 'keep_mask'+keep[i]+' = 1' 
        tst = execute(cmd) 
    END
    bad = (bad eq 1) or (keep_mask eq 0)
    FOR i=0,norder-1 DO BEGIN
        gd = where(bad[i,*] eq 0, count)
        oplot,wavl[i,gd],data[i,gd],psym=psym,color=color[i]
        if keyword_set(err) then begin
            for j = 0 , count-1 do begin
                oplot,wavl[i,gd[j]]+[0,0],$
                  data[i,gd[j]]+err[i,gd[j]]*[-1,1],color=color[i]
            end
        end
;	cmd = 'oplot,wavl'+keep[i]+',data'+keep[i]+',psym=psym,color='+color[i]
;	tst = execute(cmd)
	cmd = 'spec_w = [spec_w,reform(wavl'+keep[i]+')]'
	tst = execute(cmd)
	cmd = 'spec_d = [spec_d,reform(data'+keep[i]+')]'
	tst = execute(cmd)
    ENDFOR
    i = where(spec_w gt -1)
    spec_w = spec_w[i]
    spec_d = spec_d[i]
    data = fltarr(2,n_elements(spec_w))
    data[0,*] = spec_w
    data[1,*] = spec_d
END
